%% Fitting a transition model to the data: leaf wax stable C isotopes
%
% by Martin H. Trauth, 4 December 2020
% http://mres.uni-potsdam.de

%%
% Lupien, R., Russell, J., Yost, C., Kingston, J., Deino, A., Logan, J.,
% Schuh, A., Cohen, A. (2019) Vegetation change in the Baringo Basin, East
% Africa across the onset of Northern Hemisphere Glaciation 3.3–2.6 Ma.
% Palaeogeography Palaeoclimatology Palaeoecology,
% https://dx.doi.org/10.1016/j.palaeo.2019.109426.
clear, clc, close all
dataraw = load('data_lupien_2020.txt');

dataraw(:,1) = -dataraw(:,1);
dataraw = dataraw(dataraw(:,1)>=-3.3 & dataraw(:,1)<-2.6,:);
data.Age  = dataraw(:,1);
data.d13C = dataraw(:,2);

%%
% Nonlinear least squares fit of a sigmoid function
fo = fitoptions('Method','NonlinearLeastSquares',...
    'StartPoint',[0 0 -3.0 -2.0],...
    'Lower',[-30 -10 -4 -30],...
    'Upper',[ 30  10 -1 30]);
ft = fittype(@(a,b,c,d,x) a+b*(1./(1+exp(-d*(x-c)))),...
    'Coeff',{'a','b','c','d'},...
    'Options',fo);

%%
t = -3.5 : 0.1 : -1;
[f,fgof,fout] = fit(data.Age,data.d13C,ft)
y = f.a+f.b*(1./(1+exp(-f.d*(t-f.c))));

%%
rft = load('rampfit.dat');
rs = 'c = -3.045 (-3.431,-2.659)';

%%
xl = ['Age (Ma)'];
yl = ['\delta^{13}C_{wax} (',char(8240),')'];

%%
figure('Position',[50 1000 800 250])
axes('Position',[0.1 0.2 0.8 0.7],...
    'XLim',[-4 -1],...
    'YLim',[-30 -13],...
    'FontSize',12,...
    'Box','On'), hold on
p = plot(f,data.Age,data.d13C);
p(1).Color = [0.1 0.3 0.6];
p(1).Marker = 'Square';
p(1).MarkerSize = 6;
p(1).MarkerFaceColor = [0.1 0.3 0.6];
p(1).MarkerEdgeColor = [0.1 0.3 0.6];
p(2).Color = [0.7 0.3 0.1];
p(2).LineWidth = 1.5;
p(2).LineStyle = '-';

line(-rft(:,1),rft(:,3),...                % Ramp Fit
    'Color',[0.7 0.3 0.1],...
    'LineWidth',1.5,...
    'LineStyle','-.')

lgd = legend('Data Points','Sigmoid Fit','Ramp Fit');
lgd.Location = 'southeast';
lgd.Box = 'Off';
lgd.FontSize = 12;
lgd.AutoUpdate = 'Off';
xlabel(xl,...
    'FontSize',12)
ylabel(yl,...
    'FontSize',12)
    
print -dpng -r300 commands_lupien2020_sigmoid_ramp_vs2.png






